-- filepath: UART/UART_Image_Interface.vhd
-- =============================================================================
-- UART Image Interface for ONEAI CNN Integration
-- =============================================================================
-- This module receives 128x128 image data via UART and provides it to
-- the CNN system using the streaming interface format.
--
-- Protocol:
-- - Baud rate: 115200 (configurable)
-- - Data format: 8-bit grayscale values per pixel
-- - Frame start: 0xFF 0xAA 0x55 (3 byte header)
-- - Frame size: 128*128 = 16384 bytes + 3 header bytes = 16387 total
-- =============================================================================

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.numeric_std.all;
use work.CNN_Config_Package.all;
use work.CNN_Data_Package.all;

entity UART_Image_Interface is
    generic (
        CLK_FREQ    : natural := 100_000_000;  -- System clock frequency in Hz
        BAUD_RATE   : natural := 3_000_000;--115200;      -- UART baud rate
        FRAME_WIDTH : natural := 128;         -- Image width
        FRAME_HEIGHT: natural := 128          -- Image height
    );
    port (
        clk         : in std_logic;
        reset_n     : in std_logic;
        
        -- UART interface
        uart_rx     : in std_logic;
        uart_tx     : out std_logic;
        
        -- CNN streaming interface (output to CNN)
        rxStream  : out CNN_Stream_T;
        rxData    : out CNN_Values_T(2 downto 0);
        
        -- CNN streaming interface (input from CNN)
        txStream  : in CNN_Stream_T;
        txData    : in CNN_Values_T(0 downto 0)
    );
end UART_Image_Interface;


architecture rtl of UART_Image_Interface is

    -- UART components
    component UART_RX_COMP is
        generic (
            CLK_FREQ  : natural := CLK_FREQ;
            BAUD_RATE : natural := BAUD_RATE
        );
        port (
            clk       : in std_logic;
            reset_n   : in std_logic;
            rx        : in std_logic;
            rx_data   : out std_logic_vector(7 downto 0);
            rx_valid  : out std_logic
        );
    end component;

    component UART_TX_COMP is
        generic (
            CLK_FREQ  : natural := CLK_FREQ;
            BAUD_RATE : natural := BAUD_RATE
        );
        port (
            clk       : in std_logic;
            reset_n   : in std_logic;
            tx        : out std_logic;
            tx_data   : in std_logic_vector(7 downto 0);
            tx_start  : in std_logic;
            tx_busy   : out std_logic
        );
    end component;

    -- Constants
    constant FRAME_SIZE : natural := FRAME_WIDTH * FRAME_HEIGHT;
    constant TOTAL_BYTES : natural := FRAME_SIZE; -- no RGB
    constant HEADER_BYTE_1 : std_logic_vector(7 downto 0) := x"FF";
    constant HEADER_BYTE_2 : std_logic_vector(7 downto 0) := x"AA";
    constant HEADER_BYTE_3 : std_logic_vector(7 downto 0) := x"55";
    constant ECHO_BYTE : std_logic_vector(7 downto 0) := x"24";
    
    -- State machine
    type state_type is (
        IDLE_STATE,
        WAIT_HEADER_1,
        WAIT_HEADER_2,
        WAIT_HEADER_3,
        RECEIVE_PIXEL_DATA,
        OUTPUT_PIXEL,
        FRAME_COMPLETE,
        ERROR_STATE
    );
    signal state : state_type := IDLE_STATE;
    
    -- UART signals
    signal rx_data : std_logic_vector(7 downto 0);
    signal rx_valid : std_logic;
    signal tx_data : std_logic_vector(7 downto 0);
    signal tx_start : std_logic := '0';
    signal tx_busy : std_logic;
    
    -- Image buffer
    signal pixel_buffer : std_logic_vector(7 downto 0) := (others => '0');
    
    -- Counters and position tracking
    signal byte_counter : natural range 0 to TOTAL_BYTES := 0;
    signal pixel_counter : natural range 0 to FRAME_SIZE := 0;
    
    -- Pixel position calculation
    signal current_column : natural range 0 to FRAME_WIDTH-1 := 0;
    signal current_row : natural range 0 to FRAME_HEIGHT-1 := 0;
    signal last_column : natural range 0 to FRAME_WIDTH-1 := 0;
    signal last_row : natural range 0 to FRAME_HEIGHT-1 := 0;
    
    -- Output control
    signal output_active : std_logic := '0';
    signal rxStream_Reg  : CNN_Stream_t;
    signal rxData_Reg    : CNN_Values_T(2 downto 0);
    signal pixel_ready : std_logic := '0';
    
    -- Status signals
	signal echo : std_logic := '0';

begin

    -- UART RX instance
    uart_rx_inst : UART_RX_COMP
    generic map (
        CLK_FREQ  => CLK_FREQ,
        BAUD_RATE => BAUD_RATE
    )
    port map (
        clk       => clk,
        reset_n   => reset_n,
        rx        => uart_rx,
        rx_data   => rx_data,
        rx_valid  => rx_valid
    );

    -- UART TX instance (for acknowledgments/status)
    uart_tx_inst : UART_TX_COMP
    generic map (
        CLK_FREQ  => CLK_FREQ,
        BAUD_RATE => BAUD_RATE
    )
    port map (
        clk       => clk,
        reset_n   => reset_n,
        tx        => uart_tx,
        tx_data   => tx_data,
        tx_start  => tx_start,
        tx_busy   => tx_busy
    );

    -- Main state machine
    process(clk)
    begin
        if rising_edge(clk) then
            if reset_n = '0' then
                state <= IDLE_STATE;
                byte_counter <= 0;
                pixel_counter <= 0;
                output_active <= '0';
                pixel_buffer <= (others => '0');
				echo <= '0';
            else
                -- Default values
                pixel_ready <= '0';
				echo <= '0';
                
                case state is
                    when IDLE_STATE =>
                        byte_counter <= 0;
                        pixel_counter <= 0;
                        output_active <= '0';
                        
                        if rx_valid = '1' then
                            if rx_data = HEADER_BYTE_1 then
                                state <= WAIT_HEADER_2;
                            else
								if rx_data = ECHO_BYTE then
									echo <= '1';
								end if;
                                   state <= IDLE_STATE;
                            end if;
                        end if;
                    
                    when WAIT_HEADER_2 =>
                        if rx_valid = '1' then
                            if rx_data = HEADER_BYTE_2 then
                                state <= WAIT_HEADER_3;
                            else
                                state <= IDLE_STATE;
                            end if;
                        end if;
                    
                    when WAIT_HEADER_3 =>
                        if rx_valid = '1' then
                            if rx_data = HEADER_BYTE_3 then
                                state <= RECEIVE_PIXEL_DATA;
                            else
                                state <= IDLE_STATE;
                            end if;
                        end if;
                    
                    when RECEIVE_PIXEL_DATA =>
                        if rx_valid = '1' then
						
							pixel_buffer <= rx_data;
							pixel_ready <= '1';
                            
                            -- Update counters
                            byte_counter <= byte_counter + 1;
							
							-- Update column and row
							last_column <= current_column;
							last_row <= current_row;
							if current_column = FRAME_WIDTH-1 then
								current_column <= 0;
								if current_row < FRAME_HEIGHT-1 then
									current_row <= current_row + 1;
								end if;
							else
								current_column <= current_column + 1;
							end if;
                            
                            -- Check if frame complete
                            if byte_counter >= TOTAL_BYTES - 1 then
                                state <= FRAME_COMPLETE;
                                output_active <= '1';
                                current_column <= 0;
                                current_row <= 0;
                            end if;
                        end if;
                    
                    when FRAME_COMPLETE =>
                        state <= IDLE_STATE;
                    
                    when others =>
                        state <= IDLE_STATE;
                end case;
            end if;

            -- output stream data
            rxStream.Data_Valid <= rxStream_Reg.Data_Valid;
            rxStream.Column <= rxStream_Reg.Column;
            rxStream.Row <= rxStream_Reg.Row;
            rxStream.Filter <= rxStream_Reg.Filter;
            rxData <= rxData_Reg;
        end if;
    end process;

    -- CNN streaming output
    rxStream_Reg.Data_CLK <= clk;
    rxStream.Data_CLK <= clk;
    rxStream_Reg.Data_Valid <= pixel_ready;
    rxStream_Reg.Column <= last_column;
    rxStream_Reg.Row <= last_row;
    rxStream_Reg.Filter <= 0;

    -- Convert 8-bit RGB to 7-bit for CNN (shift right by 1)
    rxData_Reg(0) <= to_integer(unsigned(pixel_buffer(7 downto 0))); -- R (7-bit)
    rxData_Reg(1) <= to_integer(unsigned(pixel_buffer(7 downto 0)));  -- G (7-bit)
    rxData_Reg(2) <= to_integer(unsigned(pixel_buffer(7 downto 0)));   -- B (7-bit)
	
	
	-- TX process
	process(clk)
    begin
        if rising_edge(clk) then
            if reset_n = '0' then
                tx_start <= '0';
				tx_data <= x"00";
            else
                -- Default values
                tx_start <= '0';
				
				-- transmit CNN output
				if txStream.Data_Valid = '1' then
					tx_data <= std_logic_vector(to_unsigned(txData(0),8));
					tx_start <= '1';
				else
					-- transmit echo reply
					if echo = '1' then
						tx_data <= x"42";
						tx_start <= '1';
					end if;
				end if;
				
            end if;
        end if;
    end process;

end rtl;