/*
ONEAI CNN with UART Interface for Agilex 3 FPGA
===============================================
This top-level design integrates:
- UART Interface for receiving 128x128 RGB image data
- ONEAI CNN for image processing/classification
- Status LEDs and control interface
- Clock management and reset handling

UART Protocol:
- Baud Rate: 115200
- Frame Format: Header (0xFF 0xAA 0x55) + 16784 pixel bytes
- Response: Byte with class identified by CNN.

Pin Assignments:
- UART RX/TX: Connect to external UART interface
- LEDs: Status indication
- Push Button: Manual reset/control
*/

module cnn_fpga_demo (

    // Reset & Clock
    input  wire        io96_3a_pb1_fpga_rst_n, // resetn & pb1
    input  wire        hvio_pllrefclk,         // 100MHz on HVIO

    // GPIO                 
    input  wire        io96_3a_pb0,            // Push button 0
    output wire        io96_3a_led0,           // Status LED 0

    // UART Interface
    input  wire        uart_rx_pin,                // UART receive
    output wire        uart_tx_pin                // UART transmit
	
);

    // =================================================================
    // Internal Signals
    // =================================================================
    
    // Clock and Reset Management
    wire        system_clk;           // Main system clock (100MHz)
    wire        system_reset_n;       // Synchronized reset
    reg  [7:0]  reset_sync = 8'h00;   // Reset synchronizer
    
    // UART Interface Signals
    wire        uart_frame_ready;
    wire        uart_error_flag;
    wire [15:0] uart_bytes_received;
    wire [3:0]  uart_current_state;
	wire		uart_rx;
	wire		uart_tx;
    
    // CNN Interface Signals (UART -> CNN)
    wire        cnn_stream_clk;
    wire        cnn_stream_valid;
    wire [6:0]  cnn_stream_column;
    wire [6:0]  cnn_stream_row;
    wire [31:0] cnn_stream_filter;
    wire [7:0] cnn_input_data_r;
    wire [7:0] cnn_input_data_g;
    wire [7:0] cnn_input_data_b;
    
    // CNN Output Signals
    wire        cnn_output_clk;
    wire        cnn_output_valid;
    wire [6:0]  cnn_output_column;
    wire [6:0]  cnn_output_row;
    wire [31:0] cnn_output_filter;
    wire [31:0] cnn_output_data;
    wire [31:0] cnn_output_cycle;
    
    // Status and Control
    reg  [25:0] led_counter = 26'h0;  // LED blink counter
    wire        heartbeat_led;        // Heartbeat indicator
    wire        manual_reset;         // Manual reset from button
    
    // =================================================================
    // Clock and Reset Management
    // =================================================================
    
    // Use 100MHz clock as main system clock
    assign system_clk = hvio_pllrefclk;
    
    // Reset synchronizer
    always @(posedge system_clk or negedge io96_3a_pb1_fpga_rst_n) begin
        if (!io96_3a_pb1_fpga_rst_n) begin
            reset_sync <= 8'h00;
        end else begin
            reset_sync <= {reset_sync[6:0], 1'b1};
        end
    end
    
    // Manual reset from push button (active low, debounced)
    reg [19:0] button_debounce = 20'h0;
    reg button_sync = 1'b1;
    
    always @(posedge system_clk) begin
        if (io96_3a_pb0 == 1'b0) begin
            if (button_debounce < 20'hFFFFF) begin
                button_debounce <= button_debounce + 1'b1;
            end else begin
                button_sync <= 1'b0;
            end
        end else begin
            button_debounce <= 20'h0;
            button_sync <= 1'b1;
        end
    end
    
    assign manual_reset = button_sync;
    assign system_reset_n = reset_sync[7] & manual_reset;
    
    // =================================================================
    // UART Image Interface Instance
    // =================================================================
    
    UART_Image_Interface uart_inst (
        // Clock and Reset
        .clk                (system_clk),
        .reset_n            (system_reset_n),
        
        // UART Physical Interface
        .uart_rx            (uart_rx),
        .uart_tx            (uart_tx),
        
        // CNN Streaming Interface Output
        .rxStream            ({cnn_stream_clk, cnn_stream_valid, cnn_stream_column, 
                             cnn_stream_row, cnn_stream_filter}),
        .rxData              ({cnn_input_data_r, cnn_input_data_g, cnn_input_data_b}),
        
        // CNN Streaming Interface Input
        .txStream            ({cnn_output_clk, cnn_output_valid, cnn_output_column, 
                             cnn_output_row, cnn_output_filter}),
        .txData              (cnn_output_data)
    );
    
    // =================================================================
    // ONEAI CNN Instance
    // =================================================================
    
    CNN oneai_cnn_inst (
        // Input Stream from UART
        .iStream            ({cnn_stream_clk, cnn_stream_valid, cnn_stream_column,
                             cnn_stream_row, cnn_stream_filter}),
        .iData_1            ({cnn_input_data_r, cnn_input_data_g, cnn_input_data_b}),
        
        // Output Stream
        .oStream_1          ({cnn_output_clk, cnn_output_valid, cnn_output_column,
                             cnn_output_row, cnn_output_filter}),
        .oData_1            (cnn_output_data),
        .oCycle_1           (cnn_output_cycle)
    );
    
    // =================================================================
    // Status and LED Control
    // =================================================================
    
    // Heartbeat counter for LED blinking
    always @(posedge system_clk or negedge system_reset_n) begin
        if (!system_reset_n) begin
            led_counter <= 26'h0;
        end else begin
            led_counter <= led_counter + 1'b1;
        end
    end
    
    // Heartbeat LED (blinks at ~1Hz when system is running)
    assign heartbeat_led = led_counter[25];  // ~0.75Hz blink rate at 100MHz
    
    // LED assignments
    assign io96_3a_led0 = heartbeat_led;     // Heartbeat - system alive
    
	
    // =================================================================
    // UART PIN assignments
    // =================================================================
	
	assign uart_tx_pin = uart_tx;
	assign uart_rx = uart_rx_pin;

endmodule